import { state, dom } from './state.js';
import * as api from './api.js';
import * as ui from './ui.js';
import * as input from './input.js';

let cursorHideTimeout = null;
const HIDE_DELAY = 3000;

function hideCursor() {
    document.body.style.cursor = 'none';
}

document.addEventListener('mousemove', () => {
    document.body.style.cursor = 'default';
    clearTimeout(cursorHideTimeout);
    cursorHideTimeout = setTimeout(hideCursor, HIDE_DELAY);
});

hideCursor();

async function initializeRenderer() {
    try {
        console.log("Renderer: Requesting initial data...");
        
        const initialDataPromise = api.getInitialData();
        const minTimePromise = new Promise(resolve => setTimeout(resolve, 2000));
        
        const [initialData] = await Promise.all([initialDataPromise, minTimePromise]);

        if (!initialData) {
            throw new Error("Failed to receive initial data from main process.");
        }
        console.log("Renderer: Initial data received.");
        
        state.paths = initialData.paths;
        state.availableConsoles = initialData.consoleFolders;
        state.favoriteGames = initialData.favoriteGames;
        state.allGames = initialData.gamesByConsole;
        state.arcadeTitles = initialData.arcadeTitles;

        dom.gameScreenshotImg.src = state.paths.welcomeImageUrl;
        dom.gameScreenshotImg.alt = 'Welcome';
        document.querySelector('.console-logo').src = state.paths.logoUrl;
        document.querySelector('.game-scanlines-overlay').src = state.paths.scanlinesUrl;
        document.querySelector('.bottom-right-buttons-image').src = state.paths.buttonsUrl;
        dom.suspendButton.querySelector('img').src = state.paths.powerIcons.suspend;
        dom.shutdownButton.querySelector('img').src = state.paths.powerIcons.shutdown;
        dom.restartButton.querySelector('img').src = state.paths.powerIcons.restart;
        dom.exitAppButton.querySelector('img').src = state.paths.powerIcons.exit;
        
        api.setupApiListeners();
        ui.setupClickListeners();
        input.setupInputHandlers();
        
        state.supportPopupButtons = [dom.supportPopupPatreonButton, dom.supportPopupCloseButton];
        state.playDownloadMenuButtons = [dom.playButton, dom.playOnlineButton, dom.downloadOptionsButton];
        state.saveManageMenuButtons = [dom.saveAppButton, dom.saveComputerButton, dom.addToQueueButton, dom.manageSavedButton];
        state.savePromptButtons = [dom.savePromptYesButton, dom.savePromptNoButton];
        state.tabMenuButtons = [dom.changeConsoleButton, dom.showLobbyButton, dom.showFavoritesButton, dom.showDownloadedButton, dom.searchGameButton, dom.editUsernameButton, dom.relayConfigButton, dom.powerOptionsButton, dom.quitAppButton];
        state.powerMenuButtons = [dom.suspendButton, dom.shutdownButton, dom.restartButton, dom.exitAppButton];

        ui.renderConsoleLists(state.availableConsoles);

        ui.closeVersionMenu();
        ui.closeTabMenu();
        ui.hideSupportPopup();
        ui.closePlayDownloadMenu();
        ui.closeSaveManageMenu();
        ui.resetProgressOverlayState();
        ui.hideSavePrompt();
        ui.closeLobbyMenu();
        ui.closeVirtualKeyboard();
        ui.closePowerMenu();
        ui.hideNetplayAlert();
        
        api.fetchNotification(); 

        setTimeout(() => {
            ui.showInitialSelectionScreen();
            api.scanLobbyAndShowAlert();
        }, 10);
        
        console.log("Renderer fully initialized.");

        document.body.classList.add('loaded');

    } catch(error) {
        console.error("A fatal error occurred during renderer initialization:", error);
        document.body.innerHTML = `<div style="color:red;padding:20px; font-family: monospace;"><h1>Error Initializing</h1><p>${error.message}</p><pre>${error.stack}</pre></div>`;
        document.body.classList.add('loaded');
    }
}

document.addEventListener('DOMContentLoaded', initializeRenderer);